<?php
/**
 * Fingerprint Viewer - Toplanan verileri görüntüle
 * 
 * Kullanım: yourdomain.com/_viewer.php?key=YOUR_SECRET_KEY
 * 
 * Özellikler:
 * - User-Agent listesi ve sayıları
 * - Header pattern'leri
 * - URL/Query string analizi
 * - Referer kaynakları
 * - Request timing analizi
 * - Raw request örnekleri
 * - JSON export
 */

// Secret key kontrolü
define('SECRET_KEY', 'change_this_secret_key_123'); // _collector.php ile aynı olmalı
define('DATA_DIR', __DIR__ . '/data');
define('DB_FILE', DATA_DIR . '/fingerprints.db');

// Doğrulama
$providedKey = $_GET['key'] ?? '';
if ($providedKey !== SECRET_KEY) {
    http_response_code(404);
    die('404 Not Found');
}

// Export modu
$exportMode = isset($_GET['export']);
$exportType = $_GET['export'] ?? '';

// Veritabanı bağlantısı
if (!file_exists(DB_FILE)) {
    die('Henüz veri yok. Birkaç istek geldikten sonra tekrar deneyin.');
}

$db = new PDO('sqlite:' . DB_FILE);
$db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

// JSON Export
if ($exportMode) {
    header('Content-Type: application/json; charset=utf-8');
    $exportData = [];

    switch ($exportType) {
        case 'user_agents':
            $stmt = $db->query("SELECT * FROM user_agents ORDER BY count DESC");
            $exportData = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
        case 'headers':
            $stmt = $db->query("SELECT * FROM header_stats ORDER BY header_name, count DESC");
            $exportData = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
        case 'patterns':
            $stmt = $db->query("SELECT * FROM header_order_patterns ORDER BY count DESC");
            $exportData = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
        case 'urls':
            $stmt = $db->query("SELECT * FROM url_patterns ORDER BY count DESC");
            $exportData = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
        case 'referers':
            $stmt = $db->query("SELECT * FROM referers ORDER BY count DESC");
            $exportData = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
        case 'raw':
            $stmt = $db->query("SELECT * FROM raw_requests ORDER BY id DESC LIMIT 100");
            $exportData = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
        case 'all':
            $exportData = [
                'user_agents' => $db->query("SELECT * FROM user_agents ORDER BY count DESC")->fetchAll(PDO::FETCH_ASSOC),
                'header_stats' => $db->query("SELECT * FROM header_stats ORDER BY header_name, count DESC")->fetchAll(PDO::FETCH_ASSOC),
                'header_patterns' => $db->query("SELECT * FROM header_order_patterns ORDER BY count DESC")->fetchAll(PDO::FETCH_ASSOC),
                'urls' => $db->query("SELECT * FROM url_patterns ORDER BY count DESC LIMIT 1000")->fetchAll(PDO::FETCH_ASSOC),
                'referers' => $db->query("SELECT * FROM referers ORDER BY count DESC")->fetchAll(PDO::FETCH_ASSOC),
                'raw_samples' => $db->query("SELECT * FROM raw_requests ORDER BY id DESC LIMIT 50")->fetchAll(PDO::FETCH_ASSOC),
                'summary' => getSummary($db)
            ];
            break;
        default:
            $exportData = ['error' => 'Invalid export type. Use: user_agents, headers, patterns, urls, referers, raw, all'];
    }

    echo json_encode($exportData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// Özet istatistikler
function getSummary($db)
{
    return [
        'total_requests' => $db->query("SELECT SUM(count) FROM user_agents")->fetchColumn() ?: 0,
        'unique_user_agents' => $db->query("SELECT COUNT(*) FROM user_agents")->fetchColumn(),
        'unique_header_patterns' => $db->query("SELECT COUNT(*) FROM header_order_patterns")->fetchColumn(),
        'unique_referers' => $db->query("SELECT COUNT(*) FROM referers")->fetchColumn(),
        'unique_urls' => $db->query("SELECT COUNT(*) FROM url_patterns")->fetchColumn(),
        'first_request' => $db->query("SELECT MIN(first_seen) FROM user_agents")->fetchColumn(),
        'last_request' => $db->query("SELECT MAX(last_seen) FROM user_agents")->fetchColumn(),
    ];
}

$summary = getSummary($db);

// Verileri al
$userAgents = $db->query("SELECT * FROM user_agents ORDER BY count DESC LIMIT 100")->fetchAll(PDO::FETCH_ASSOC);
$headerPatterns = $db->query("SELECT * FROM header_order_patterns ORDER BY count DESC LIMIT 50")->fetchAll(PDO::FETCH_ASSOC);
$referers = $db->query("SELECT * FROM referers ORDER BY count DESC LIMIT 50")->fetchAll(PDO::FETCH_ASSOC);
$urls = $db->query("SELECT * FROM url_patterns ORDER BY count DESC LIMIT 100")->fetchAll(PDO::FETCH_ASSOC);
$rawSamples = $db->query("SELECT * FROM raw_requests ORDER BY id DESC LIMIT 20")->fetchAll(PDO::FETCH_ASSOC);

// Accept-Language değerleri
$acceptLangs = $db->query("SELECT header_value, count FROM header_stats WHERE header_name = 'ACCEPT-LANGUAGE' ORDER BY count DESC LIMIT 20")->fetchAll(PDO::FETCH_ASSOC);

// Accept değerleri
$accepts = $db->query("SELECT header_value, count FROM header_stats WHERE header_name = 'ACCEPT' ORDER BY count DESC LIMIT 20")->fetchAll(PDO::FETCH_ASSOC);

// Accept-Encoding değerleri
$acceptEncodings = $db->query("SELECT header_value, count FROM header_stats WHERE header_name = 'ACCEPT-ENCODING' ORDER BY count DESC LIMIT 20")->fetchAll(PDO::FETCH_ASSOC);

// Sec-Fetch değerleri
$secFetchSite = $db->query("SELECT header_value, count FROM header_stats WHERE header_name = 'SEC-FETCH-SITE' ORDER BY count DESC LIMIT 10")->fetchAll(PDO::FETCH_ASSOC);
$secFetchMode = $db->query("SELECT header_value, count FROM header_stats WHERE header_name = 'SEC-FETCH-MODE' ORDER BY count DESC LIMIT 10")->fetchAll(PDO::FETCH_ASSOC);
$secFetchDest = $db->query("SELECT header_value, count FROM header_stats WHERE header_name = 'SEC-FETCH-DEST' ORDER BY count DESC LIMIT 10")->fetchAll(PDO::FETCH_ASSOC);

// Connection values
$connections = $db->query("SELECT header_value, count FROM header_stats WHERE header_name = 'CONNECTION' ORDER BY count DESC LIMIT 10")->fetchAll(PDO::FETCH_ASSOC);

// Timing analizi - son 5 dakika
$fiveMinAgo = microtime(true) - 300;
$requestRate = $db->query("SELECT COUNT(*) FROM request_timing WHERE timestamp > $fiveMinAgo")->fetchColumn();
$avgRatePerMin = round($requestRate / 5, 1);

?>
<!DOCTYPE html>
<html lang="tr">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🔍 Fingerprint Analysis</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', system-ui, sans-serif;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            color: #e0e0e0;
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
        }

        h1 {
            text-align: center;
            margin-bottom: 30px;
            color: #00d9ff;
            text-shadow: 0 0 20px rgba(0, 217, 255, 0.3);
        }

        .summary-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 30px;
        }

        .summary-card {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 12px;
            padding: 20px;
            text-align: center;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .summary-card .number {
            font-size: 2.5em;
            font-weight: bold;
            color: #00d9ff;
        }

        .summary-card .label {
            color: #888;
            font-size: 0.9em;
            margin-top: 5px;
        }

        .section {
            background: rgba(255, 255, 255, 0.03);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid rgba(255, 255, 255, 0.08);
        }

        .section h2 {
            color: #00d9ff;
            margin-bottom: 15px;
            font-size: 1.3em;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            padding-bottom: 10px;
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        th,
        td {
            padding: 10px 12px;
            text-align: left;
            border-bottom: 1px solid rgba(255, 255, 255, 0.05);
        }

        th {
            background: rgba(0, 217, 255, 0.1);
            color: #00d9ff;
            font-weight: 600;
        }

        tr:hover {
            background: rgba(255, 255, 255, 0.03);
        }

        .count-badge {
            background: rgba(0, 217, 255, 0.2);
            color: #00d9ff;
            padding: 3px 10px;
            border-radius: 20px;
            font-size: 0.85em;
            font-weight: bold;
        }

        .ua-text {
            font-family: 'Consolas', monospace;
            font-size: 0.85em;
            word-break: break-all;
            color: #aaa;
        }

        .header-order {
            font-family: 'Consolas', monospace;
            font-size: 0.8em;
            color: #7ec8e3;
            word-break: break-all;
        }

        .export-links {
            text-align: center;
            margin-bottom: 20px;
        }

        .export-links a {
            color: #00d9ff;
            text-decoration: none;
            margin: 0 15px;
            padding: 8px 16px;
            background: rgba(0, 217, 255, 0.1);
            border-radius: 6px;
            transition: all 0.3s;
        }

        .export-links a:hover {
            background: rgba(0, 217, 255, 0.2);
        }

        .raw-json {
            background: #0a0a15;
            border-radius: 8px;
            padding: 15px;
            font-family: 'Consolas', monospace;
            font-size: 0.8em;
            overflow-x: auto;
            white-space: pre-wrap;
            max-height: 300px;
            overflow-y: auto;
            color: #7ec8e3;
            border: 1px solid rgba(255, 255, 255, 0.1);
            margin-bottom: 15px;
        }

        .grid-2 {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }

        @media (max-width: 900px) {
            .grid-2 {
                grid-template-columns: 1fr;
            }
        }

        .highlight {
            color: #ffcc00;
        }

        .referer-analysis {
            color: #90EE90;
        }

        .timestamp {
            color: #888;
            font-size: 0.85em;
        }
    </style>
</head>

<body>
    <div class="container">
        <h1>🔍 Bot Fingerprint Intelligence</h1>

        <!-- Summary -->
        <div class="summary-grid">
            <div class="summary-card">
                <div class="number">
                    <?= number_format($summary['total_requests']) ?>
                </div>
                <div class="label">Toplam İstek</div>
            </div>
            <div class="summary-card">
                <div class="number">
                    <?= $summary['unique_user_agents'] ?>
                </div>
                <div class="label">Unique User-Agent</div>
            </div>
            <div class="summary-card">
                <div class="number">
                    <?= $summary['unique_header_patterns'] ?>
                </div>
                <div class="label">Header Pattern</div>
            </div>
            <div class="summary-card">
                <div class="number">
                    <?= $summary['unique_referers'] ?>
                </div>
                <div class="label">Unique Referer</div>
            </div>
            <div class="summary-card">
                <div class="number">
                    <?= $avgRatePerMin ?>/dk
                </div>
                <div class="label">Son 5dk Ort. Rate</div>
            </div>
        </div>

        <!-- Export Links -->
        <div class="export-links">
            📥 Export:
            <a href="?key=<?= SECRET_KEY ?>&export=all">Tümü (JSON)</a>
            <a href="?key=<?= SECRET_KEY ?>&export=user_agents">User-Agents</a>
            <a href="?key=<?= SECRET_KEY ?>&export=headers">Headers</a>
            <a href="?key=<?= SECRET_KEY ?>&export=patterns">Patterns</a>
            <a href="?key=<?= SECRET_KEY ?>&export=raw">Raw Samples</a>
        </div>

        <!-- User Agents -->
        <div class="section">
            <h2>🤖 User-Agent Listesi (
                <?= count($userAgents) ?> unique)
            </h2>
            <table>
                <tr>
                    <th style="width:80px">Sayı</th>
                    <th>User-Agent</th>
                    <th style="width:180px">İlk / Son Görülme</th>
                </tr>
                <?php foreach ($userAgents as $ua): ?>
                    <tr>
                        <td><span class="count-badge">
                                <?= number_format($ua['count']) ?>
                            </span></td>
                        <td class="ua-text">
                            <?= htmlspecialchars($ua['user_agent']) ?>
                        </td>
                        <td class="timestamp">
                            <?= $ua['first_seen'] ?><br>
                            <?= $ua['last_seen'] ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>
        </div>

        <div class="grid-2">
            <!-- Accept-Language -->
            <div class="section">
                <h2>🌍 Accept-Language</h2>
                <table>
                    <tr>
                        <th>Değer</th>
                        <th>Sayı</th>
                    </tr>
                    <?php foreach ($acceptLangs as $row): ?>
                        <tr>
                            <td class="ua-text">
                                <?= htmlspecialchars($row['header_value']) ?>
                            </td>
                            <td><span class="count-badge">
                                    <?= number_format($row['count']) ?>
                                </span></td>
                        </tr>
                    <?php endforeach; ?>
                </table>
            </div>

            <!-- Accept-Encoding -->
            <div class="section">
                <h2>📦 Accept-Encoding</h2>
                <table>
                    <tr>
                        <th>Değer</th>
                        <th>Sayı</th>
                    </tr>
                    <?php foreach ($acceptEncodings as $row): ?>
                        <tr>
                            <td class="ua-text">
                                <?= htmlspecialchars($row['header_value']) ?>
                            </td>
                            <td><span class="count-badge">
                                    <?= number_format($row['count']) ?>
                                </span></td>
                        </tr>
                    <?php endforeach; ?>
                </table>
            </div>
        </div>

        <div class="grid-2">
            <!-- Sec-Fetch Headers -->
            <div class="section">
                <h2>🔒 Sec-Fetch Headers</h2>
                <h4 style="color:#888;margin-bottom:10px">Sec-Fetch-Site:</h4>
                <?php foreach ($secFetchSite as $row): ?>
                    <span class="count-badge" style="margin-right:10px">
                        <?= htmlspecialchars($row['header_value']) ?>:
                        <?= $row['count'] ?>
                    </span>
                <?php endforeach; ?>

                <h4 style="color:#888;margin:15px 0 10px">Sec-Fetch-Mode:</h4>
                <?php foreach ($secFetchMode as $row): ?>
                    <span class="count-badge" style="margin-right:10px">
                        <?= htmlspecialchars($row['header_value']) ?>:
                        <?= $row['count'] ?>
                    </span>
                <?php endforeach; ?>

                <h4 style="color:#888;margin:15px 0 10px">Sec-Fetch-Dest:</h4>
                <?php foreach ($secFetchDest as $row): ?>
                    <span class="count-badge" style="margin-right:10px">
                        <?= htmlspecialchars($row['header_value']) ?>:
                        <?= $row['count'] ?>
                    </span>
                <?php endforeach; ?>
            </div>

            <!-- Connection -->
            <div class="section">
                <h2>🔗 Connection Header</h2>
                <table>
                    <tr>
                        <th>Değer</th>
                        <th>Sayı</th>
                    </tr>
                    <?php foreach ($connections as $row): ?>
                        <tr>
                            <td class="ua-text">
                                <?= htmlspecialchars($row['header_value']) ?>
                            </td>
                            <td><span class="count-badge">
                                    <?= number_format($row['count']) ?>
                                </span></td>
                        </tr>
                    <?php endforeach; ?>
                </table>
            </div>
        </div>

        <!-- Header Order Patterns -->
        <div class="section">
            <h2>📋 Header Sıralama Pattern'leri</h2>
            <p style="color:#888;margin-bottom:15px;font-size:0.9em">
                Header sırası, gerçek tarayıcı mı yoksa bot mu olduğunu anlamak için kritik bir sinyal.
            </p>
            <table>
                <tr>
                    <th style="width:80px">Sayı</th>
                    <th>Header Sıralaması</th>
                </tr>
                <?php foreach ($headerPatterns as $pattern): ?>
                    <tr>
                        <td><span class="count-badge">
                                <?= number_format($pattern['count']) ?>
                            </span></td>
                        <td class="header-order">
                            <?= htmlspecialchars($pattern['header_order']) ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>
        </div>

        <!-- Referers -->
        <div class="section">
            <h2>🔗 Referer Kaynakları</h2>
            <table>
                <tr>
                    <th style="width:80px">Sayı</th>
                    <th>Referer</th>
                </tr>
                <?php foreach ($referers as $ref): ?>
                    <tr>
                        <td><span class="count-badge">
                                <?= number_format($ref['count']) ?>
                            </span></td>
                        <td class="referer-analysis">
                            <?= htmlspecialchars(substr($ref['referer'], 0, 200)) ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>
        </div>

        <!-- URL Patterns -->
        <div class="section">
            <h2>🌐 URL Pattern'leri (Query String dahil)</h2>
            <table>
                <tr>
                    <th style="width:80px">Sayı</th>
                    <th>URL</th>
                </tr>
                <?php foreach (array_slice($urls, 0, 30) as $url): ?>
                    <tr>
                        <td><span class="count-badge">
                                <?= number_format($url['count']) ?>
                            </span></td>
                        <td class="ua-text">
                            <?= htmlspecialchars($url['full_url']) ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>
        </div>

        <!-- Raw Request Samples -->
        <div class="section">
            <h2>📝 Son İstek Örnekleri (Raw Data)</h2>
            <?php foreach (array_slice($rawSamples, 0, 5) as $sample): ?>
                <div class="raw-json">
                    <strong>⏰
                        <?= $sample['timestamp'] ?>
                    </strong> |
                    <?= $sample['method'] ?> |
                    <?= $sample['protocol'] ?>
                    <strong>URL:</strong>
                    <?= htmlspecialchars($sample['full_url']) ?>
                    <strong>User-Agent:</strong>
                    <?= htmlspecialchars($sample['user_agent']) ?>
                    <strong>Header Order:</strong>
                    <?= htmlspecialchars($sample['header_order']) ?>
                    <strong>All Headers:</strong>
                    <?= htmlspecialchars($sample['all_headers']) ?>
                </div>
            <?php endforeach; ?>
        </div>

        <p style="text-align:center;color:#555;margin-top:30px;font-size:0.85em">
            🛡️ Fingerprint Collector | Son güncelleme:
            <?= date('Y-m-d H:i:s') ?>
        </p>
    </div>
</body>

</html>